
package salesmansolver;

import java.util.*;
import java.util.Map.Entry;

/**
 *
 * @author Andrea Barbagallo
 */
public class Population {
    // A collection of possible routes (Chromosomes)
    
    //                 < Fitness , Chromosome >
    private final TreeMap<Double, Chromosome> population;
    private final Integer maxPop;
    private Integer eliteSize;
    private double currentGenBest;
    
    public Population(Integer maxPopulation) {
        population = new TreeMap<>();
        currentGenBest = 0d;
        this.maxPop = maxPopulation;
        
        if (maxPop <= 0)
            System.err.println("ALERT: negative (or zero) max population size!");
    }
    
    public Integer getSize() { return population.size(); }
    
    public Chromosome randomChromosome(LinkedHashMap<Integer, Gene> cityList) {
        
        Chromosome random = new Chromosome();

        List<Integer> keys = new ArrayList(cityList.keySet());

        Collections.shuffle(keys);

        for (Integer o : keys)
            random.add(cityList.get(o));
        
        return random;
    }
    
    public void initialPopulation(LinkedHashMap<Integer, Gene> cityList) {
        currentGenBest = 0d;
        
        if (population.isEmpty()) {
            for (int i = 0; i < maxPop; i++) {
                Chromosome tmp = randomChromosome(cityList);
                population.put(tmp.getFitness(), tmp);
                
                if (currentGenBest < tmp.getFitness())
                    currentGenBest = tmp.getFitness();
            }
            
            System.err.println("Initial Population created! (" + population.size() + "/" + maxPop + " elements)");
        } else {
            System.err.println("Error: Initial Population called on a not empty population!");
        }
    }
    
    public void nextGeneration(Integer selectionSize, Integer eliteSize, double mutationRate) {
        this.eliteSize = eliteSize;
        currentGenBest = 0d;
        setNewPopulation(mutation(recombination(selection(selectionSize)), mutationRate));
    }
    
    public Chromosome bestPath() {
        return population.get(population.lastKey());
    }
    
    public Chromosome worstPath() {
        return population.get(population.firstKey());
    }
    
    public void print() {
        System.out.println(population);
    }
    
    public int size() {
        return population.size();
    }
    
    // Elitism Chromosomes are excluded
    public double getCurrentGenerationBestPath() {
        return population.get(currentGenBest).getRouteDistance();
    }
    
    /**
     * The selection is made using the "Fitness proportionate selection", so
     * the probability of selecting an individual is based on it's fitness.
     * This method is also called "roulette wheel selection". In addition to
     * that, due to the elitism selection, the best routes will be kept.
     * 
     * @param selectionSize How many Chromosomes should be selected
     * @param eliteSize How many Chromosomes are choosen using elitism
     * @return The selected individuals ordered by fitness
     */
    private TreeMap<Double, Chromosome> selection(Integer selectionSize) {
        eliteSize = Integer.min(eliteSize, population.size());
        TreeMap<Double, Chromosome> selected = new TreeMap<>();
        
        // Elitism selection
        for (int i = 0; i < eliteSize; i++) {
            Map.Entry<Double, Chromosome> elite = population.lastEntry();
            selected.put(elite.getKey(), elite.getValue());
            population.remove(elite.getKey());
        }
        
        if (population.isEmpty()) {
            return selected;
        }
        
        // Roulette wheel selection
        final double multiplier = population.lastKey();
        double randomDouble;
        double selectedKey;
        
        Random r = new Random();
        
        for (int i = 0; i < (selectionSize-eliteSize); i++) {
            randomDouble = r.nextDouble()*multiplier;
            selectedKey = population.ceilingKey(randomDouble);
            
            selected.put(selectedKey, population.get(selectedKey));
        }
        
        if (selected.size() > maxPop)
            System.err.println("ALERT: selected size greater then max population!");
        
        return selected;
    }
    
    /**
     * The recombination is made using the partially mapped crossover (PMX)
     * in order to mantain the order of Genes as much as possible.
     * @param selected
     * @return 
     */
    private TreeMap<Double, Chromosome> recombination(TreeMap<Double, Chromosome> selected) {
        TreeMap<Double, Chromosome> recombined = new TreeMap<>();
        
        ArrayList<Chromosome> all = new ArrayList<>(selected.values());
            
        for (int j = 0; j < all.size(); j++) {
            Chromosome child = combine(all.get(all.size()-j-1), all.get(j));
            recombined.put(child.getFitness(), child);

            if (currentGenBest < child.getFitness())
                currentGenBest = child.getFitness();
        }
        
        recombined.putAll(selected);
        
        return recombined;
    }
    
    /**
     * Combine two parents, creating a child using the partially mapped
     * crossover (PMX).
     * 
     * @param p1 Parent n1
     * @param p2 Parent n2
     * @return Child
     */
    private Chromosome combine(Chromosome p1, Chromosome p2) {
        Chromosome childChromosome = new Chromosome();
        Random r = new Random();
        
        int t1 = r.nextInt(p1.size());
        int t2 = r.nextInt(p1.size());
        
        Integer start = Integer.min(t1, t2);
        Integer end = Integer.max(t1, t2);
        
        Gene[] child = new Gene[p1.size()];
        
        Stack<Integer> toSolve = new Stack<>();
        
        for (int i = 0; i < p2.size(); i++)
            child[i] = p2.get(i);
        
        for (int i = start; i <= end; i++) {
            child[i] = p1.get(i);
        }
        
        for (int i = start; i <= end; i++) {
            for (int j = start; j <= end; j++) {
                if (p2.get(i).equals(child[j]))
                    break;
                
                if (j == end) {
                    toSolve.push(i);
                }
            }
        }
        
        // Solve problems
        boolean solved;
        
        while (!toSolve.empty()) {
            Integer tmp = toSolve.pop();
            
            Gene i = p2.get(tmp);
            Gene j = child[tmp];
            
            solved = false;
            
            while (!solved) {
                
                for (int c = 0; c < p2.size(); c++) {
                    if (p2.get(c).equals(j)) {
                        if (c >= start && c <= end) {
                            j = p1.get(c);
                            break;
                        } else {
                            child[c] = i;
                            solved = true;
                            break;
                        }
                    }
                }
            }
        }
        
        for (int i = 0; i < child.length; i++)
            childChromosome.add(new Gene(child[i]));
        
        return childChromosome;
    }
    
    private TreeMap<Double, Chromosome> mutation(TreeMap<Double, Chromosome> recombined, double rate) {
        TreeMap<Double, Chromosome> mutated = new TreeMap<>();
        Random r = new Random();
        
        for (Chromosome c: recombined.values()) {
            
            if (r.nextDouble() <= rate) {
                Chromosome mutation = new Chromosome(c);

                int t1 = r.nextInt(mutation.size());
                int t2 = r.nextInt(mutation.size());
                
                Integer start = Integer.min(t1, t2);
                Integer end = Integer.max(t1, t2);
                
                while (start < end) {
                    mutation.swap(start++, end--);
                }

                if (currentGenBest < mutation.getFitness())
                    currentGenBest = mutation.getFitness();
                
                mutated.put(mutation.getFitness(), mutation);
            }
        }
        
        recombined.putAll(mutated);
        
        return recombined;
    }
    
    private void setNewPopulation(TreeMap<Double, Chromosome> mutated) {
        population.clear();
        for (int i = Integer.min(maxPop, mutated.size()); i > 0; i--) {
            Entry<Double, Chromosome> tmp = mutated.lastEntry();
            mutated.remove(tmp.getKey());
            population.put(tmp.getKey(), tmp.getValue());
        }
        
    }
}
